/*
 *
 * Iskra SISTEMI (C) 2003  DM (davor.munda@kopica-sp.si)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 *
 * Module: rtx.h - RTX Interface Header.
 *
 * Version: 1.1.0 - 2003/04/01 : Basic version - DM
 * Version: 1.2.0 - 2003/04/14 : The same API for LXRT and kernel Hard RT, added alloc routines - DM
 * Version: 2.1.0 - 2003/05/06 : Separate RTX modules (cpp class usage in LXRT UserSpace) - DM
 * Version: 2.2.0 - 2003/05/20 : Remove timers interface (we use only one global timer) - DM
 *
 */


#ifndef _RTX_DEF_
 #define _RTX_DEF_

#ifdef __cplusplus
 extern "C" {
#endif


#include <rtai_common.h>
#include <rtx_types.h>
#include <rtx_add.h>


#ifndef __KERNEL__  /* lxrt - user space */
#ifndef __cplusplus
#error " --- Must use CPP compiler !"
#endif
#endif



/**
    Functions
**/


/** Start RT task. **/
extern RTX_RESULT  XRunTask( rtx_task_handlePtr        hndPtr,       /* handle pointer */
                             const rtx_priority        priority,     /* task priority */
                             const rtx_task_stacksize  stack,        /* stack size - bytes */
                             const rtx_task_mode       mode,         /* task mode */
                             const rtx_periodic_tick   nanoTick,     /* tick for periodic mode - nano sec */
                             rtai_task                 task_entry,   /* entry point */
                             rtx_thread_handlePtr      thread_hndPtr,/* thread handle pointer */
                             const rtx_task_arg        arg           /* argument */ );

/** Lock RT scheduler - works only in kernel space. **/
extern void  XLock( void );

/** Unock RT scheduler - works only in kernel space. **/
extern void  XUnLock( void );

/** Release MBX channel. **/
extern RTX_RESULT  XReleaseMBX( rtx_mbx_handlePtr  mbxChPtr /* channel pointer */ );

/** Stop RT task. **/
extern RTX_RESULT  XKillTask( rtx_task_handlePtr  hndPtr,   /* handle pointer */
                              rtx_thread_handle   hndThread /* thread handle */ );

/** Stop current thread/task. **/
extern void  XExit( void );

/** Get current task id. **/
extern rtx_task_handlePtr  XGetId( void );

/** Change priority. **/
extern RTX_RESULT  XCPri( rtx_task_handlePtr  hndPtr,  /* handle pointer */
                          const rtx_priority  priority /* priority */ );

/** Sleep no of ticks for periodic task. **/
extern void  XDelay( rtx_periodic_tickno         ticks,  /* no ticks */
                     const rtx_periodic_tickPtr  tickPtr /* tick pointer */ );

/** Sleep no of nano seconds. **/
extern void  XDelayShort( rtx_nano_time  ntime /* no nano secs */ );

/** Suspend thread/task. **/
extern RTX_RESULT  XStop( rtx_task_handlePtr  hndPtr /* handle pointer */ );

/** Suspend thread/task. **/
extern RTX_RESULT  XResume( rtx_task_handlePtr  hndPtr /* handle pointer */ );

/** Get nano seconds tick duration. **/
extern rtx_periodic_tick  XGetTickInterv( const rtx_periodic_tickPtr  tickPtr /* tick pointer */ );

/** Set nano seconds tick duration for periodic thread/task. **/
extern RTX_RESULT  XSetTickInterv( rtx_periodic_tickPtr     tickPtr,  /* tick pointer */
                                   const rtx_periodic_tick  nanoTick  /* tick for periodic mode - nano sec */ );

/** Get truncated no ticks from nano seconds for periodic thread/task. **/
extern rtx_periodic_tickno  XGetNoTicksTrunc( const rtx_nano_time         ntime,  /* no nano secs */
                                              const rtx_periodic_tickPtr  tickPtr /* tick pointer */ );

/** Get rounded up no ticks from nano seconds for periodic thread/task. **/
extern rtx_periodic_tickno  XGetNoTicksRoundUp( const rtx_nano_time         ntime,  /* no nano secs */
                                                const rtx_periodic_tickPtr  tickPtr /* tick pointer */ );

/** Get no needed ticks from nano seconds for periodic thread/task. **/
extern rtx_periodic_tickno  XGetNoTicks( const rtx_nano_time         ntime, /* no nano secs */
                                         const rtx_periodic_tickPtr  tickPtr /* tick pointer */ );

/** Send MBX message. **/
extern RTX_RESULT  XSendMBX( rtx_mbx_handlePtr         mbxChPtr,      /* channel pointer */
                             const rtx_mbx_msgPtr      mbxMsgPtr,     /* message pointer */
                             const rtx_mbx_msgLen      msgLen,        /* message length - bytes */
                             const rtx_nano_time       busyWait,      /* timeout if channel is bussy - nano secs - 0 don't wait */
                             const rtx_nano_time       readTaskWait,  /* timeout (after busyWait) to be read and deleted by task - nano secs - 0 don't wait */
                             const rtx_task_handlePtr  readTaskHndPtr /* read task handle pointer */ );

/** Wait MBX message - get it and put it back into the channel - FIFO order. **/
extern RTX_RESULT  XWaitMBX( rtx_mbx_handlePtr     mbxChPtr,  /* channel pointer */
                             rtx_mbx_msgPtr        mbxMsgPtr, /* message pointer */
                             const rtx_mbx_msgLen  msgLen,    /* message length - bytes */
                             const rtx_nano_time   arriveWait /* timeout if channel is empty - nano secs - 0 don't wait */ );

/** Read MBX message - delete message from the channel. **/
extern RTX_RESULT  XReadMBX( rtx_mbx_handlePtr         mbxChPtr,       /* channel pointer */
                             rtx_mbx_msgPtr            mbxMsgPtr,      /* message pointer */
                             const rtx_mbx_msgLen      msgLen,         /* message length - bytes */
                             const rtx_nano_time       arriveWait,     /* timeout if channel is empty - nano secs - 0 don't wait */
                             const rtx_task_handlePtr  replyTaskHndPtr /* reply task handle pointer - NULL don't reply */ );

/** Get free mutex **/
extern RTX_RESULT  XGetFreeMuteX( rtx_mutex_handlePtr   mxPtr /* mutex pointer */ );

/** Release mutex **/
extern RTX_RESULT  XReleaseMuteX( rtx_mutex_handlePtr   mxPtr /* mutex pointer */ );

/** Lock section - blocked wait if locked **/
extern RTX_RESULT  XLockSection( rtx_mutex_handlePtr   mxPtr /* mutex pointer */ );

/** Unlock section - always unlock **/
extern RTX_RESULT  XUnlockSection( rtx_mutex_handlePtr   mxPtr /* mutex pointer */ );

/** Try lock section - lock if unlocked, else return failure **/
extern RTX_RESULT  XTryLockSection( rtx_mutex_handlePtr   mxPtr /* mutex pointer */ );

/** Create RTF FIFO **/
extern RTX_RESULT  XRtfCreate( const rtx_fifo_id     fifoId,     /* rtf fifo id - 0,...,63  for /dev/rtf0,...,/dev/rtf63 devices */
                               const rtx_fifo_size   fifoSize,   /* rtf fifo size - bytes */
                               rtx_fifo_devhandlePtr fifoDevPtr, /* rtf fifo device pointer */
                               rtx_fifo_sizePtr      fifoSizePtr /* rtf fifo size space pointer */ );

/** Destroy RTF FIFO **/
extern void  XRtfDestroy( const rtx_fifo_id     fifoId,     /* rtf fifo id - 0,...,63  for /dev/rtf0,...,/dev/rtf63 devices */
                          rtx_fifo_devhandlePtr fifoDevPtr, /* rtf fifo device pointer */
                          rtx_fifo_sizePtr      fifoSizePtr,/* rtf fifo size space pointer */
				          rtx_thread_handlePtr  thread_hndPtr/* thread handle pointer */ );

/** Reset RTF FIFO - discard all data **/
extern RTX_RESULT  XRtfReset( const rtx_fifo_id     fifoId,     /* rtf fifo id - 0,...,63  for /dev/rtf0,...,/dev/rtf63 devices */
                              rtx_fifo_devhandlePtr fifoDevPtr, /* rtf fifo device pointer */
                              rtx_fifo_sizePtr      fifoSizePtr /* rtf fifo size space pointer */ );

/** Put RTF FIFO - write data into the fifo within RT task **/
extern rtx_fifo_okBytes  XRtfPut( const rtx_fifo_id        fifoId,    /* rtf fifo id - 0,...,63  for /dev/rtf0,...,/dev/rtf63 devices */
                                  const rtx_fifo_dataPtr   dataPtr,   /* data pointer */
                                  const rtx_fifo_dataSize  dataSize,  /* data size - bytes */
                                  rtx_fifo_devhandlePtr    fifoDevPtr /* rtf fifo device pointer */ );

/** Get RTF FIFO - read data from the fifo within RT task **/
extern rtx_fifo_okBytes  XRtfGet( const rtx_fifo_id        fifoId,    /* rtf fifo id - 0,...,63  for /dev/rtf0,...,/dev/rtf63 devices */
                                  rtx_fifo_dataPtr         dataPtr,   /* data pointer */
                                  const rtx_fifo_dataSize  dataSize,  /* data size - bytes */
                                  rtx_fifo_devhandlePtr    fifoDevPtr /* rtf fifo device pointer */ );

/** Alloc space. **/
extern void*  XMAlloc( const size_t  size /* size - bytes */ );

/** Free space. **/
extern void  XMFree( void* ptr /* allocated memory pointer */ );


#ifndef __KERNEL__  /* lxrt - user space */

#define RTX_TASK(handleSpace,handleP,entryF,tickSpace,tickP,threadHandle,thandleP)  \
    rtx_task_handlePtr    handleP = NULL;                                           \
    rtx_periodic_tick     tickSpace;                                                \
    rtx_periodic_tickPtr  tickP = &tickSpace;                                       \
    rtx_thread_handle     threadHandle;                                             \
    rtx_thread_handlePtr  thandleP = &threadHandle;                                 \
    rt_entry  entryF( rt_arg arg );

#define RTX_MBX(handleSpace,handleP)   \
    rtx_mbx_handlePtr  handleP = NULL;

#define RTX_RTFHANDLER(handleP,tickSpace,entryF,threadHandle) \
    rtx_task_handlePtr  handleP;                              \
    rtx_periodic_tick   tickSpace;                            \
    rtx_thread_handle   threadHandle;                         \
    rtf_hentry  entryF( rtf_harg  arg );

#define RTF_HANDLER_BEGIN(handleP,tickSpace,rtfId)  \
    rtx_fifo_id  fifoId = rtfId;                    \
    if (arg) {}                                     \
    if ( runThread( &handleP,                       \
                     RTX_TPRIOR_PERIODIC_NORMAL,    \
                     1024,                          \
                     RTX_TASK_PERIODIC,             \
                     10000000,                      \
                    &tickSpace ) != RTX_SUCCESS )   \
		return (void *)1;                           \
    while ( 1 ) {;

#define RTF_HANDLER_END(result,tickSpace)  \
      XDelay( 1, &tickSpace );             \
    }                                      \
    rt_end;

#define  XREGISTER(hndP,prior,stack,mode,period,tick) { \
      if (arg) {}                                  \
      if ( runThread( &hndP,                       \
                       prior,                      \
                       stack,                      \
                       mode,                       \
                       period,                     \
                      &tick ) != RTX_SUCCESS )     \
        return (void *)1;                        }

#define  XGetFreeMBX(mbxP,size,count)  getMBX( &mbxP, size, count )
#define  XRtfCreateHandler(fifoId,fifoHnd,threadHndP)  rtfCreateHnd( fifoId, fifoHnd, threadHndP )

#else   /* kernel space */

#define RTX_TASK(handleSpace,handleP,entryF,tickSpace,tickP,threadHandle,thandleP)  \
    rtx_task_handle       handleSpace;                                              \
    rtx_task_handlePtr    handleP = &handleSpace;                                   \
    rtx_periodic_tick     tickSpace;                                                \
    rtx_periodic_tickPtr  tickP = &tickSpace;                                       \
    rtx_thread_handle     threadHandle;                                             \
    rtx_thread_handlePtr  thandleP = &threadHandle;                                 \
    rt_entry  entryF( rt_arg arg );

#define RTX_MBX(handleSpace,handleP)   \
    rtx_mbx_handle     handleSpace;    \
    rtx_mbx_handlePtr  handleP = &handleSpace;

#ifndef OS_WIN32
#define RTX_RTFHANDLER(handleP,tickSpace,entryF,threadHandle)  \
    rtx_thread_handle   threadHandle;                          \
    rtf_hentry  entryF( rtf_harg  arg );
#else
#define RTX_RTFHANDLER(handleP,tickSpace,entryF,threadHandle) \
    rtx_task_handlePtr  handleP;                              \
    rtx_periodic_tick   tickSpace;                            \
    rtx_thread_handle   threadHandle;                         \
    rtf_hentry  entryF( rtf_harg  arg );
#endif

#ifndef OS_WIN32
#define RTF_HANDLER_BEGIN(handleP,tickSpace,rtfId)  \
    rtx_fifo_id  fifoId = arg;
#define RTF_HANDLER_END(result,tickSpace)     return(result);
#else
#define RTF_HANDLER_BEGIN(handleP,tickSpace,rtfId,threadHandle)  \
    rtx_fifo_id  fifoId = rtfId;                                 \
    if (arg) {}                                                  \
    while ( threadHandle ) {;
#define RTF_HANDLER_END(result,tickSpace)  \
      os_Sleep(10);                           \
    }                                      \
    os_ExitTask(0);
#endif

#define  XREGISTER(hndP,prior,stack,mode,period,tick)  { if (arg) {} }

#define  XGetFreeMBX(mbxP,size,count)  getMBX( mbxP, size, count )
#ifndef OS_WIN32
#define  XRtfCreateHandler(fifoId,fifoHnd,threadHndP)  rtfCreateHnd( fifoId, fifoHnd )
#else
#define  XRtfCreateHandler(fifoId,fifoHnd,threadHndP)  rtfCreateHnd( fifoId, fifoHnd, threadHndP )
#endif

#endif


#ifdef __cplusplus
 }
#endif

#endif  /* _RTX_DEF_ */

